#include "GLSLProgram.h"

#define glDeleteObject glDeleteObjectARB

CGLSLProgram::CGLSLProgram(SHADER_TYPE st) :
	isInstalled(false) {
	program = glCreateProgram();

	vs = glCreateShader(GL_VERTEX_SHADER);
	shaderType = st;
	switch (shaderType) {
	case VS_FS:
		fs = glCreateShader(GL_FRAGMENT_SHADER);
		break;
	case VS_GS:
		gs = glCreateShader(GL_GEOMETRY_SHADER_EXT);
		break;
	case VS:
		break;
	case VS_GS_FS:
		gs = glCreateShader(GL_GEOMETRY_SHADER_EXT);
		fs = glCreateShader(GL_FRAGMENT_SHADER);
		break;
	}

}

GLhandleARB CGLSLProgram::GetProgram() {
	return program;
}

CGLSLProgram::~CGLSLProgram() {

	glDeleteObject(vs);
	//TODO: generic delete to geometry shader attached objects
	switch (shaderType) {
	case VS_FS:
		glDeleteObject(fs);
		break;
	case VS_GS:
		glDeleteObject(gs);
		break;
	case VS:
		break;
	}
	glDeleteProgram(program);

}

std::string CGLSLProgram::OpenShaderFile(std::string filename) {
	std::string buff;
	std::ifstream file;
	std::cerr.flush();
	file.open(filename.c_str());
	std::string line;
	while (std::getline(file, line))
		buff += line + "\n";

	return buff;
}

void CGLSLProgram::CompileShaderObject(std::string source, GLhandleARB object) {
	//Source code assignment
	const GLcharARB *txt = source.c_str();
	glShaderSourceARB(object, 1, &txt, NULL);

	//Compile shader object
	glCompileShaderARB(object);

	if (!CheckShader(object)) {
		std::cout << "Compilation failed \n";
	} else {
		glAttachObjectARB(program, object);
		GLstateTest();
	}
}

GLint CGLSLProgram::CheckShader(GLhandleARB object) {
	GLint ok = 0;
	glGetObjectParameterivARB(object, GL_OBJECT_COMPILE_STATUS_ARB, &ok);
	if (!ok) {
		int maxLength=4096;
		char *infoLog = new char[maxLength];
		glGetInfoLogARB(object, maxLength, &maxLength, infoLog);
		std::cout<<"Compilation error: "<<infoLog<<"\n";
		delete []infoLog;
	}

	return ok;
}

void CGLSLProgram::InitFragmentShader(const char *filen) {
	fs_path = filen;
	CompileShaderObject(OpenShaderFile(fs_path), fs);
}

void CGLSLProgram::InitFragmentShader(const char *filen, const char *defines) {
	fs_path = filen;
	std::string txt = defines+OpenShaderFile(fs_path);
	CompileShaderObject(txt, fs);
}

void CGLSLProgram::InitVertexShader(const char *filen) {
	vs_path = filen;
	CompileShaderObject(OpenShaderFile(vs_path), vs);
}

void CGLSLProgram::InitVertexShader(const char *filen, const char *defines) {
	vs_path = filen;
	std::string txt = defines+OpenShaderFile(vs_path);
	CompileShaderObject(txt, vs);
}

void CGLSLProgram::InitGeometryShader(const char *filen, GLenum input,
		GLenum output, GLuint nPrimitivesOut) {
	gs_path = filen;

	CompileShaderObject(OpenShaderFile(gs_path), gs);

	glProgramParameteriEXT(program,GL_GEOMETRY_INPUT_TYPE_EXT , input);
	glProgramParameteriEXT(program,GL_GEOMETRY_OUTPUT_TYPE_EXT , output);

	glProgramParameteriEXT(program,GL_GEOMETRY_VERTICES_OUT_EXT, nPrimitivesOut);
}

void CGLSLProgram::InitGeometryShader(const char *filen, const char *defines,
		GLenum input, GLenum output, GLuint nPrimitivesOut) {
	gs_path = filen;

	std::string txt = defines+OpenShaderFile(gs_path);

	CompileShaderObject(txt, gs);

	glProgramParameteriEXT(program,GL_GEOMETRY_INPUT_TYPE_EXT , input);
	glProgramParameteriEXT(program,GL_GEOMETRY_OUTPUT_TYPE_EXT , output);

	glProgramParameteriEXT(program,GL_GEOMETRY_VERTICES_OUT_EXT, nPrimitivesOut);
}

void CGLSLProgram::SetUniform3f(const GLcharARB* name, const GLfloat v0,
		const GLfloat v1, const GLfloat v2) {
	Use();
	glUniform3f(glGetUniformLocation(program, name), v0, v1, v2);
}

void CGLSLProgram::SetUniform2f(const GLcharARB* name, const GLfloat v0,
		const GLfloat v1) {
	Use();
	glUniform2f(glGetUniformLocation(program, name), v0, v1);
}

void CGLSLProgram::SetUniform1f(const GLcharARB* name, const GLfloat v0) {
	Use();
	glUniform1f(glGetUniformLocation(program, name), v0);
}

void CGLSLProgram::SetUniform1i(const GLcharARB* name, const GLint v0) {
	Use();
	glUniform1i(glGetUniformLocation(program, name), v0);
}

void CGLSLProgram::SetTexture(const GLcharARB* name, const GLuint textureid,
		const GLint i) {
	Use();
	int my_sampler_uniform_location= glGetUniformLocationARB(program, name);
	glActiveTexture(GL_TEXTURE0 + i);
	glBindTexture(GL_TEXTURE_2D, textureid);
	glUniform1iARB(my_sampler_uniform_location, i);
}

void CGLSLProgram::LinkValidate() {
	glLinkProgram(program);
	glValidateProgram(program);
	GLstateTest();
	Use();
	GLstateTest();
	Stop();
}
void CGLSLProgram::Use() {
	if (!isInstalled) {
		isInstalled = true;
		GLstateTest();
		glUseProgram(program);
	}
}
void CGLSLProgram::Stop() {
	isInstalled = false;
	glUseProgram(NULL);
}